<?php

namespace App\Http\Controllers\Api;

use App\Mail\Order;
use App\Mail\Voucher;
use App\Services\PdfParser;
use App\Models\OrderMaster;
use App\Services\TempToLive;
use App\Services\PrintVoucher;
use App\Services\WebpayService;
use App\Models\OrderMasterTemp;
use App\Services\CouponService;
use App\Mail\DriverConfirmation;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Services\BookingDataService;
use Illuminate\Support\Facades\Mail;

class WebpayController extends Controller
{
    /**
     * @var string
     */
    private $expectedResponseCode = '0000';
    
    /**
     * @param WebpayService $service
     *
     * @return JsonResponse
     */
    public function getForm(WebpayService $service): JsonResponse
    {
        $forms = $service->getForms(request()->header('tok'));
        
        return response()->json([
            'data'   => $forms->data,
            'client' => $forms->client,
        ]);
    }
    
    /**
     * @param TempToLive $tempToLive
     * @param CouponService $couponService
     * @param PdfParser $pdfParser
     *
     * @return JsonResponse
     */
    public function success(TempToLive $tempToLive, CouponService $couponService, PdfParser $pdfParser): JsonResponse
    {
        if (! request()->has('response_code')) {
            return response()->json([
                'status'  => 'error',
                'message' => [
                    'title'    => __('api.paymentCancelTitle'),
                    'subtitle' => __('api.paymentCancelSubtitle'),
                ],
            ], 404);
        }
        
        if (request()->response_code != $this->expectedResponseCode) {
            return response()->json([
                'status'  => 'error',
                'message' => [
                    'title'    => __('api.paymentCancelTitle'),
                    'subtitle' => __('api.paymentCancelSubtitle'),
                ],
            ], 404);
        }
        
        if (OrderMaster::where('MCardNumber', request()->header('tok'))->exists()) {
            return response()->json([
                'status'  => 'error',
                'message' => [
                    'title'    => __('api.paymentCancelTitle'),
                    'subtitle' => __('api.paymentCancelSubtitle'),
                ],
            ], 404);
        }
        
        $transfer = OrderMasterTemp::with('details', 'details.extras')
                                   ->where('MOrderKey', request()->header('tok'))
                                   ->first();
        
        if (! $transfer) {
            return response()->json([
                'status'  => 'error',
                'message' => [
                    'title'    => __('api.paymentCancelTitle'),
                    'subtitle' => __('api.paymentCancelSubtitle'),
                ],
            ], 404);
        }
        
        $orderMaster = $tempToLive->execute();
        
        if (! $orderMaster) {
            return response()->json([
                'status'  => 'error',
                'message' => [
                    'title'    => __('api.paymentCancelTitle'),
                    'subtitle' => __('api.paymentCancelSubtitle'),
                ],
            ], 404);
        }
        
        if (request()->coupon_code) {
            $couponService->increment(request()->coupon_code);
        }
        
        $voucher = PrintVoucher::execute($orderMaster->MOrderID);
        
        $pdfParser->execute($voucher, $orderMaster->MOrderID);
        
        $this->sendEmails($orderMaster, $transfer, $voucher);
        
        return response()->json([
            'status'  => 'success',
            'message' => [
                'title'    => __('api.paymentSuccessTitle'),
                'subtitle' => $transfer->MPaymentMethod == '2' ? __('api.bookingSuccessCashSubtitle') : __('api.paymentSuccessSubtitle'),
            ],
        ]);
    }
    
    /**
     * @param $orderMaster
     * @param $transfer
     * @param $voucher
     */
    private function sendEmails($orderMaster, $transfer, $voucher)
    {
        // ako je cash onda je temp order, trazi se potvrda
        if ($orderMaster->MPaymentMethod == '2') {
            Mail::to($transfer->MPaxEmail)
                ->bcc('cms@jamtransfer.com')
                ->send(new Voucher($transfer, $orderMaster->MOrderID));
            
            return;
        }
        // ako je placeno online, onda nije temp order nego prava rezervacija
        // ne trazi se potvrda, nego se salje voucher
        Mail::to($orderMaster->MPaxEmail)
            ->bcc([
                'cms@jamtransfer.com',
                'officejamgroup@gmail.com',
            ])
            ->send(new Order($orderMaster, $voucher));
        
        Mail::to([
            $orderMaster->details->last()->DriverEmail,
        ])
            ->bcc('cms@jamtransfer.com')
            ->send(new DriverConfirmation($orderMaster));
    }
}
