<?php

namespace App\Services;

use App\Models\Place;
use App\Models\Route;
use Illuminate\Support\Str;
use App\Models\DriverRoute;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;

class SearchPlaceService
{
    /**
     * @var Place
     */
    protected $place;
    
    public function __construct(Place $place)
    {
        $this->place = $place;
    }
    
    public function searchPickup(string $query)
    {
        $routes = $this->getRoutes();
        
        return Place::search($query)
                    ->get()
                    ->filter(function ($place) use ($routes) {
                        return in_array($place->PlaceID, $routes['from']) || in_array($place->PlaceID, $routes['to']);
                    })
                    ->map(function ($place) {
                        return [
                            'id'   => $place->PlaceID,
                            'name' => $this->formatName($place),
                            'type' => $place->PlaceType,
                        ];
                    })
                    ->sortBy('name')
                    ->values();
    }
    
    public function searchDropoff(int $id, string $query)
    {
        $routes = $this->getRoutes($id);
        
        return Place::search($query)
                    ->where(function ($qry) use ($routes) {
                        $qry->whereIn('PlaceID', $routes['from'])
                            ->orWhereIn('PlaceID', $routes['to']);
                    })
                    ->get()
                    ->filter(function ($place) use ($id) {
                        return $place->PlaceID != $id;
                    })
                    ->transform(function ($place) {
                        return [
                            'id'   => $place->PlaceID,
                            'name' => $this->formatName($place),
                            'type' => $place->PlaceType,
                        ];
                    })
                    ->sortBy('name')
                    ->values();
    }
    
    public static function findAvailablePlaces($query)
    {
        return Place::with('country')
                    ->where('PlaceName'.Str::upper(app()->getLocale()), 'LIKE', "%{$query}%")
                    ->active()
                    ->where(function ($query) {
                        $query->where('PlaceType', 1)
                              ->orWhere('PlaceType', 3)
                              ->orWhere('PlaceType', 6)
                              ->orWhere('PlaceType', 8)
                              ->orWhere('PlaceType', 9);
                    })
                    ->get()
                    ->filter(function ($place) {
                        return DriverRoute::where('FromID', $place->PlaceID)
                                          ->orWhere('ToID', $place->PlaceID)
                                          ->exists();
                    })
                    ->map(function ($place, $key) {
                        return self::formatPlace($place);
                    })
                    ->sortBy('name');
    }
    
    public function findAvailableDestinations($place): array
    {
        $placeId = $place->PlaceID;
        
        $data = [];
        
        $driverRoutes = DriverRoute::select('RouteID', 'FromID', 'ToID')
                                   ->distinct()
                                   ->where(function ($query) use ($placeId) {
                                       $query->where('FromID', $placeId)
                                             ->orWhere('ToID', $placeId);
                                   })
                                   ->get();
        
        foreach ($driverRoutes as $route) {
            $places = Place::where('PlaceID', $route->FromID)
                           ->orWhere('PlaceID', $route->ToID)
                           ->active()
                           ->get();
            
            foreach ($places as $place) {
                if ($place->PlaceActive == '1') {
                    # Do not repeat the From name
                    if ($place->PlaceID != $placeId and $place->PlaceActive == '1') {
                        # Add Place to array
                        $pnLang = 'PlaceName' . Lang();
                        if (strlen($pnLang) == 9) {
                            $pnLang = 'PlaceNameEN';
                        }
                    
                        $placeName = mb_strtolower($place->$pnLang, 'UTF-8');
                    
                        // fix ako nema jezika
                        if (empty($placeName)) {
                            $pnLang = 'PlaceNameEN';
                        }
                        $placeName = mb_strtolower($place->$pnLang, 'UTF-8');
                    
                        $placeName .= ', ' . getPlaceCountryCode($place->PlaceCountry);
                        # Add Place to array
                        if (! in_array($placeName . '|' . $place->PlaceNameSEO, $data)) {
                            //$fromPlaces[$place->PlaceID] = trim($placeName) . '|' . trim($place->PlaceNameSEO);
                            $data[trim($placeName)] = [
                                'ID'           => $place->PlaceID,
                                'PlaceNameSEO' => trim($place->PlaceNameSEO),
                                'PlaceType'    => $place->PlaceType
                            ];
                        }
                    }
                }
            }
        }
    
        ksort($data);
    
        $res = [];
    
        foreach ($data as $key => $value) {
            $res[] = [
                'name' => mb_strtoupper($key, 'UTF-8'),
                'id'    => $value['ID'],
                'SEO'   => $value['PlaceNameSEO'],
                'type'  => $value['PlaceType'],
        
            ];
        }
    
        return $res;
    }
    
    private static function formatPlace($place)
    {
        $placeName = Str::lower($place->{'PlaceName'.Str::upper(app()->getLocale())});
        
        return [
            'name' => Str::upper("{$placeName}, {$place->country->CountryCode}"),
            'SEO'  => trim($place->PlaceNameSEO),
            'type' => $place->PlaceType
        ];
    }
    
    private function getRoutes($id = null)
    {
        $routes = [];
        
        $driverRoutes = DB::table('v4_Routes')
                          ->select('FromID', 'ToID')
                          ->when($id, function ($query, $id) {
                              return $query->where('FromID', $id)
                                           ->orWhere('ToID', $id);
                          })
                          ->distinct()
                          ->get();
        
        foreach ($driverRoutes as $route) {
            $routes['from'][] = $route->FromID;
            $routes['to'][] = $route->ToID;
        }
        
        return $routes;
    }
    
    private function formatName($place): string
    {
        $countryCode = $place->country ? $place->country->CountryCode : '';
        
        return Str::upper((! empty($place->{'PlaceName' . Lang()}) ? $place->{'PlaceName' . Lang()} : $place->{'PlaceNameEN'}) . ', ' . $countryCode);
    }
    
    public function findPlaceFromPlaceName($name)
    {
        $place = Place::where('PlaceNameSEO', 'like', $name)
                      ->orWhere('PlaceNameEN', 'like', $name)
                      ->orWhere('PlaceNameRU', 'like', $name)
                      ->orWhere('PlaceNameDE', 'like', $name)
                      ->orWhere('PlaceNameFR', 'like', $name)
                      ->orWhere('PlaceNameIT', 'like', $name)
                      ->first();
        
        if (!$place) {
            return null;
        }
    
        return [
            'id'   => $place->PlaceID,
            'name' => $this->formatName($place),
            'image' => $place->Image ? config('app.wis_url') . $place->Image : config('app.wis_url') . DB::table('v4_HpImages')->find(1)->Image,
        ];
    }
    
    /**
     * @return Collection
     */
    public function getPlacesGroupedByCountry(): Collection
    {
        $routes = $this->getRoutes();
    
        return Place::select('PlaceID', 'PlaceCountry', 'PlaceName' . Lang(), 'PlaceNameSEO')
                     ->with([
                         'country' => function ($query) {
                             $query->select('CountryID', 'CountryName' . Lang());
                         },
                     ])
                     ->active()
                     ->where('PlaceType', 1)
                     ->get()
                     ->filter(function ($place) use ($routes) {
                         return in_array($place->PlaceID, $routes['from']) || in_array($place->PlaceID, $routes['to']);
                     })
                     ->groupBy('country.CountryName' . Lang())
                     ->sortKeys()
                     ->map(function ($items, $key) {
                         return [
                             'items'   => $items->map(function ($item) {
                                 return [
                                     'id'   => $item['PlaceID'],
                                     'name' => $item['PlaceName' . Lang()],
                                     'slug' => trim(stripslashes($item['PlaceNameSEO'])),
                                 ];
                             }),
                             'isShown' => false,
                         ];
                     });
    }
    
    public function getDestinations(string $type, int $id)
    {
        if ($type === 'main') {
            return $this->getMainRoutes($id);
        } else if ($type === 'top') {
            return $this->getTopRoutes($id);
        } else if ($type === 'other') {
            return $this->getOtherRoutes($id);
        }
        
        return [];
    }
    
    /**
     * @param int $id
     *
     * @return mixed
     */
    private function getMainRoutes(int $id)
    {
        return Route::select('RouteName' . Lang() . ' as name', 'p.PlaceNameSEO as slug')
                    ->join('v4_Places as p', 'PlaceID', 'ToID')
                    ->where('FromID', $id)
                    ->where('Approved', 1)
                    ->whereIn('RouteID', function ($query) {
                        $query->select('TopRouteID')->from('v4_TopRoutes')->where('Main', 1);
                    })
                    ->get();
    }
    
    /**
     * @param int $id
     *
     * @return mixed
     */
    private function getTopRoutes(int $id)
    {
        return Route::select('RouteName' . Lang() . ' as name', 'p.PlaceNameSEO as slug')
                    ->join('v4_Places as p', 'PlaceID', 'ToID')
                    ->where('FromID', $id)
                    ->where('Approved', 1)
                    ->whereIn('RouteID', function ($query) {
                        $query->select('TopRouteID')->from('v4_TopRoutes')->where('Main', 0);
                    })
                    ->get();
    }
    
    private function getOtherRoutes(int $id)
    {
        return Route::select('RouteName' . Lang() . ' as name', 'p.PlaceNameSEO as slug')
                    ->join('v4_Places as p', 'PlaceID', 'ToID')
                    ->where('FromID', $id)
                    ->where('Approved', 1)
                    ->whereNotIn('RouteID', function ($query) {
                        $query->select('TopRouteID')->from('v4_TopRoutes');
                    })
                    ->get();
    }
}
