<?php

namespace App\Services;

use App\Models\OrderMasterTemp;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\Eloquent\Model;

class WebpayService
{
    private $key;
    
    private $submit_url;
    
    private $token;
    
    private $url_components;
    
    private $transfer;
    
    /**
     *  Webpay Service.
     */
    public function __construct()
    {
        if (config('webteh.debug')) { // test payment gateway
            $this->key = config('webteh.test.key');
            $this->submit_url = config('webteh.test.url');
            $this->token = config('webteh.test.token');
            $this->url_components = config('webteh.test.url_components');
        
        } else { // LIVE payment gateway
            $this->key = config('webteh.production.key');
            $this->submit_url = config('webteh.production.url');
            $this->token = config('webteh.production.token');
            $this->url_components = config('webteh.production.url_components');
        }
    }
    
    public function getForms(string $tok)
    {
        $this->transfer = OrderMasterTemp::with('details')
                                         ->where('MOrderKey', $tok)
                                         ->first();
        
        throw_if(!$this->transfer, new \Exception('Transfer not found.'));
        
        $result = new \stdClass();
        $result->data = $this->getFormForRedirectToPaymentGateway();
        $result->client = $this->getClientSecret();
        
        return $result;
    }
    
    /**
     * Get form for redirect.
     *
     * @return array
     */
    public function getFormForRedirectToPaymentGateway(): array
    {
        $amount = $this->getAmount();
        
        return [
            'form' => [
                'ch_full_name' => $this->transfer->MPaxFirstName . ' ' . $this->transfer->MPaxLastName,
                'ch_address' => 'Address',
                'ch_city' => 'City',
                'ch_zip' => 'Zip',
                'ch_country' => $this->transfer->MCardCountry,
                'ch_phone' => $this->transfer->MPaxTel,
                'ch_email' => $this->transfer->MPaxEmail,
                'order_info' => 'Taxi Transfer',
                'amount' => $amount,
                'order_number' => $this->transfer->MOrderKey,
                'currency' => 'EUR',
                'transaction_type' => 'purchase',
                'number_of_installments' => '',
                'cc_type_for_installments' => '',
                'moto' => 'false',
                'authenticity_token' => $this->token,
                'digest' => SHA1($this->key . $this->transfer->MOrderKey . $amount . 'EUR'),
                'language' => 'en',
                'custom_params' => 'https://jamtransfer.com',
            ],
            'url' => $this->submit_url . '/form',
            'status' => true
        ];
    }
    
    /**
     * Get client secret for form.
     *
     * @return string|null
     */
    public function getClientSecret(): ?string
    {
        return $this->fetch(
            $this->prepareData()
        );
    }
    
    /**
     * Prepare data for webpay.
     *
     * @return array
     */
    private function prepareData(): array
    {
        return [
            'amount'                    => $this->getAmount(),
            'order_number'              => $this->transfer->MOrderKey,
            'currency'                  => 'EUR',
            'transaction_type'          => 'purchase',
            'order_info'                => 'Taxi Transfer',
            'scenario'                  => 'charge',
            'supported_payment_methods' => ['card'],
        ];
    }
    
    /**
     * @param $data
     *
     * @return string|null
     */
    private function fetch($data): ?string
    {
        $body_as_string = json_encode($data); // use php's standard library equivalent if Json::encode is not available in your code
        $ch = curl_init($this->url_components . '/v2/payment/new');
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");
        curl_setopt($ch, CURLOPT_POSTFIELDS, $body_as_string);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    
        $authenticity_token = $this->token;
        $body_as_string = json_encode($data);
        $timestamp = time();
        $digest = hash('sha512', $this->key . $timestamp . $authenticity_token . $body_as_string);
        $authorization = "WP3-v2 $authenticity_token $timestamp $digest";
    
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json',
                'Content-Length: ' . strlen($body_as_string),
                'Authorization: ' . $authorization,
            ]
        );
    
        $result = curl_exec($ch);
    
        if (curl_errno($ch)) {
            Log::alert('Error', curl_error($ch));
        }
    
        curl_close($ch);
    
        $response = json_decode($result);
        
        Log::info($result);
        
        if ($response->status === 'approved') {
            return $response->client_secret;
        }
        
        if ($response->status === 'declined') {
            return null;
        }
        
        return null;
    }
    
    /**
     * @return string
     */
    private function getAmount(): string
    {
        $details = $this->transfer->details;
        
        return $details->count() == 2
            ? number_format(number_format($details->first()->PayNow + $details->last()->PayNow, 2, '.', '') * 100, 0, '.', '')
            : number_format(number_format($details->first()->PayNow, 2, '.', '') * 100, 0, '.', '');
    }
    
    /**
     * Calculate prices.
     *
     * @param $transfer
     *
     * @return array
     */
    public function calculate($transfer): array
    {
        $returnTransfer = count($transfer) > 2 ? 1 : 0;
        
        $TPayNow = 0;
        $TPayLater = 0;
        $TXPayNow = 0;
        $TXPayLater = 0;
        $FullPayment = 0;
        
        // transfer znaci temp, X je za return transfer
        // first transfer Gde ide extras, na online ili cash ? Sada ja na Cash
        $TPayNow = $transfer[0]['DetailPrice'] - $transfer[0]['DriversPrice'];
        $TPayLater = $transfer[0]['DetailPrice'] - $TPayNow;
        $TPayNow=$TPayNow+$transfer[0]['ExtraCharge']; //ako ide na online
        
        if($returnTransfer) {
            $TXPayNow = $transfer[1]['DetailPrice'] - $transfer[1]['DriversPrice'];
            $TXPayLater = $transfer[1]['DetailPrice'] - $TXPayNow;
            $TPayNow=$TPayNow+$transfer[1]['ExtraCharge']; //ako ide na online
        }
        
        $FullPayment = nf($TPayNow + $TPayLater + $TXPayNow + $TXPayLater);
        
        return [$FullPayment, $TPayNow, $TPayLater, $TXPayNow, $TXPayLater];
    }
}